/*
Copyright @ 1999-2002, The Institute for Genomic Research (TIGR).  
All rights reserved.

This software is provided "AS IS".  TIGR makes no warranties, express
or implied, including no representation or warranty with respect to
the performance of the software and derivatives or their safety,
effectiveness, or commercial viability.  TIGR does not warrant the
merchantability or fitness of the software and derivatives for any
particular purpose, or that they may be exploited without infringing
the copyrights, patent rights or property rights of others. TIGR shall
not be liable for any claim, demand or action for any loss, harm,
illness or other damage or injury arising from access to or use of the
software or associated information, including without limitation any
direct, indirect, incidental, exemplary, special or consequential
damages.

This software program may not be sold, leased, transferred, exported
or otherwise disclaimed to anyone, in whole or in part, without the
prior written consent of TIGR.
*/
/*
 * $Id: StringSplitter.java,v 1.2 2002/04/05 10:54:15 lexa Exp $
 *
 * Created 10/26/2001
 *
 * Description: The class allows an application to break a string into tokens.
 *
 * Copyright (C) 2002 TIGR
 *
 */
package org.tigr.midas.util;

import java.util.NoSuchElementException;

public class StringSplitter {

    private String str;
    private char delimiter;

    private int curPosition;
    private int maxPosition;

    /**
     * Constructs a <code>StringSplitter</code> with specified delimiter.
     */
    public StringSplitter(char delimiter) {
        init("", delimiter);
    }

    public StringSplitter(String str, char delimiter) {
        init(str, delimiter);
    }
    
    /**
     * Initializes this <code>StringSplitter</code> with a new data.
     */
    public void init(String str) {
        init(str, this.delimiter);
    }

    /**
     * Initializes this <code>StringSplitter</code> with a new data and delimeter.
     */
    private void init(String str, char delimiter) {
        this.str = str;
        this.delimiter = delimiter;
        curPosition = 0;
        maxPosition = str.length();
    }

    /**
     * Skips ahead from startPos and returns the index of the next delimiter
     * character encountered, or maxPosition if no such delimiter is found.
     */
    private int scanToken(int startPos) {
        int position = startPos;
        while (position < maxPosition) {
            char c = str.charAt(position);
            if (c == delimiter)
                break;
            position++;
        }
        return position;
    }

    /**
     * Tests if there are more tokens available from this tokenizer's string. 
     * If this method returns <tt>true</tt>, then a subsequent call to 
     * <tt>nextToken</tt> with no argument will successfully return a token.
     *
     * @return  <code>true</code> if and only if there is at least one token 
     *          in the string after the current position; <code>false</code> 
     *          otherwise.
     */
    public boolean hasMoreTokens() {
        return curPosition < maxPosition;
    }

    /**
     * Returns the next token from this string tokenizer.
     *
     * @return     the next token from this string tokenizer.
     * @exception  NoSuchElementException if there are no more tokens in this
     *             tokenizer's string.
     */
    public String nextToken() {
        if (curPosition >= maxPosition)
            throw new NoSuchElementException("There are no more tokens!");
        int start = curPosition;
        curPosition = scanToken(curPosition);
        String result = str.substring(start, curPosition);
        curPosition++;
        return result;
    }

    /**
     * Returns the number of tokens.
     */
    public int countTokens() {
        int count = 0;
        int pos = 0;
        while (pos < this.maxPosition) {
            if (str.charAt(pos) == this.delimiter)
                count++;
            pos++;
        }
        return count;
    }

    /**
     *  Passes the given count of tokens.
     */
    public final void passTokens(int count) {
        for (int i = count; --i >= 0;) {
            curPosition = scanToken(curPosition);
            curPosition++;
        }
    }

    /**
     * Returns the next token value as an integer.
     */
    public final int nextIntToken() {
        return nextIntToken(0);
    }

    /**
     * Returns the next token value as an integer with specifed default value.
     */
    public final int nextIntToken(int defValue) {
        String token = nextToken();
        try {
            return Integer.parseInt(token);
        } catch (NumberFormatException e) {
            return defValue;
        }
    }

    /**
     * Returns the next token value as a long.
     */
    public final long nextLongToken() {
        return nextLongToken(0);
    }

    /**
     * Returns the next token value as a double rounded to long.
     */
    public final long nextLongToken(boolean rounded) {
        return Math.round(nextDoubleToken(0));
    }

    /**
     * Returns the next token value as a long with specifed default value.
     */
    public final long nextLongToken(long defValue) {
        String token = nextToken();
        try {
            return Long.parseLong(token);
        } catch (NumberFormatException e) {
            return defValue;
        }
    }

    /**
     * Returns the next token value as a double with specifed default value.
     */
    public final double nextDoubleToken(double defValue) {
        String token = nextToken();
        try {
            return Double.parseDouble(token);
        } catch (NumberFormatException e) {
            return defValue;
        }
    }

    /**
     * Returns the next token value as a float with specifed default value.
     */
    public final float nextFloatToken(float defValue) {
        String token = nextToken();
        try {
            return Float.parseFloat(token);
        } catch (NumberFormatException e) {
            return defValue;
        }
    }
}
