/*
 * Copyright @ 2001-2003, The Institute for Genomic Research (TIGR).
 * All rights reserved.
 *
 *
 * RatioStats.java
 *
 * Created on December 17, 2002, 11:15 AM
 * @author  wliang
 * @version
 */

package org.tigr.midas.engine;

import java.io.*;
import java.util.*;
import java.math.*;
import java.lang.*;
// Mod for MeV import org.tigr.midas.display.*;
import org.tigr.midas.math.*;
import org.tigr.midas.util.*;
import org.tigr.midas.Constant;

public class RatioStats extends Module{
    
    //From Chen, et.al's paper, 1997, J. Biomed. Optics 2(4), 364-374
    private static final float[] meanCoef = {1.001f, -0.0427f, 1.279f, 0.364f};
    private static final float[] stdCoef = {0.00225f, 1.341f, 0.190f, 6.259f};
    private static final float[] low95Coef = {0.979f, -2.706f, 2.911f, -2.805f};
    private static final float[] high95Coef = {0.989f, 3.082f, -2.830f, 28.664f};
    private static final float[] low99Coef = {0.9968f, -3.496f, 4.462f, -5.002f};
    private static final float[] high99Coef = {0.9648f, 4.810f, -15.161f, 78.349f};
    
    private ColumnWorker fCW;
    //private int totalRowCount;
    private float[] cy3Column;
    private float[] cy5Column;
    
    private float calcMean;
    private float plainMean;
    private float cVal;
    private float calcSD;
    private float high99;
    private float low99;
    private float high95;
    private float low95;
    private int validRatioCounts;
    private int confidentCounts;
    private int iterCount;
   
    public RatioStats(ColumnWorker fileCW, boolean confidenceChk, int pctConfidence, String ref) throws Exception {
        
        super();
        
        double mLimit = 1E-10;
        int maxIterations = 100;
        
        boolean[] outputFlag;
        //Mod for MeV
        //FileBrowser projFolderFB = new FileBrowser(proj.getProjPath(), Constant.READ_FILE);
        //String iterLinRegFB = projFolderFB.createSubDir("ratio_stats");
        
        
        try{
            fCW = fileCW;
            
            cy3Column = fileCW.getColumnOneArray();
            cy5Column = fileCW.getColumnTwoArray();
            
            boolean[] tempFlag = fileCW.getRowFlagArray();
            outputFlag = new boolean[tempFlag.length];
            System.arraycopy(tempFlag, 0, outputFlag, 0, tempFlag.length);
            
            //Output pre-normalization RI results
            //Mod for MeV  
            DataPrep dataPrep;
            //DataReporter dr;
            //String prePrcName = super.proj.getProjPath() + "ratio_stats" + FileBrowser.fsep + fileCW.getFileName() + "_pre.prc";
            dataPrep = new DataPrep(fileCW);
            //dr = new DataReporter(prePrcName, dataPrep.getLogProd(), dataPrep.getLogRatio(), new IndexAdjuster().addOne(dataPrep.getRawNdx()));
        
        
            float[] ratios = new float[fCW.getRowCounts()];
            float oldCalcMean = 1.0f;
            float ratioSum = 0;
            plainMean = 0;
            calcMean = 1.0f;
            validRatioCounts = 0;
            confidentCounts = 0;
            
            int totalRowCount = fCW.getRowCounts();
            
            //Compute initial valid ratios
            for (int rawNdx = 0; rawNdx < fCW.getRowCounts(); rawNdx++){
                if (cy3Column[rawNdx] > 1 && cy5Column[rawNdx] > 1 && outputFlag[rawNdx]){
                    ratios[rawNdx] = cy5Column[rawNdx] / cy3Column[rawNdx];
                    ratioSum += ratios[rawNdx];
                    validRatioCounts++;
                }else{
                    ratios[rawNdx] = 0;
                    outputFlag[rawNdx] = false;
                }
            }

            System.out.println("    Valid Ratio Counts = " + validRatioCounts);
            
            //Iteration for good calcMean
            iterCount = 1;
            
            while(true){
                //System.out.println("    Iteration " + iterCount);
                float accumulator = 0;
                plainMean = ratioSum / validRatioCounts / oldCalcMean;
                ratioSum = 0;
                
                //for (int i = 0; i < validRatioCounts; i++){
                for (int i = 0; i < fCW.getRowCounts(); i++){
                    if (outputFlag[i]){
                        ratios[i] /= plainMean;
                        accumulator += MathTools.sqr(ratios[i] - 1) / (1 + MathTools.sqr(ratios[i]));
                        ratioSum += ratios[i];
                    }else{}
                }
                
                //Compute Maximum-Likelyhood Estimator
                cVal = (float)Math.sqrt(accumulator / validRatioCounts);
                
                //Compute calcMean
                calcMean = (meanCoef[3] * cVal * cVal * cVal) + (meanCoef[2] * cVal * cVal) + (meanCoef[1] * cVal) + meanCoef[0];
                System.out.println("    ---- Iteration " + iterCount + ", plainMean = " + plainMean + ", calcMean = " + calcMean);
                
                if ( (Math.abs(calcMean - oldCalcMean) < mLimit) || (iterCount >= maxIterations) ){
                    break;
                }else{
                    oldCalcMean = calcMean;
                    iterCount++;
                }
            }
            
            //Compute calcSD
            calcSD = (stdCoef[3] * cVal * cVal * cVal) + (stdCoef[2] * cVal * cVal) + (stdCoef[1] * cVal) + stdCoef[0];
            //Comput high99
            high99 = (high99Coef[3] * cVal * cVal * cVal) + (high99Coef[2] * cVal * cVal) + (high99Coef[1] * cVal) + high99Coef[0];
            high99 *= plainMean;
            //Compute low99
            low99 = (low99Coef[3] * cVal * cVal * cVal) + (low99Coef[2] * cVal * cVal) + (low99Coef[1] * cVal) + low99Coef[0];
            low99 *= plainMean;
            //Compute high95
            high95 = (high95Coef[3] * cVal * cVal * cVal) + (high95Coef[2] * cVal * cVal) + (high95Coef[1] * cVal) + high95Coef[0];
            high95 *= plainMean;
            //Compute low95
            low95 = (low95Coef[3] * cVal * cVal * cVal) + (low95Coef[2] * cVal * cVal) + (low95Coef[1] * cVal) + low95Coef[0];
            low95 *= plainMean;
            
            System.out.println("    ratio mean = " + plainMean);
            System.out.println("    ratio function mean = " + calcMean);
            System.out.println("    ratio density SD = " + calcSD);
            System.out.println("    maximum-likelyhood estimator = " + cVal);
            System.out.println("    95% confidence lower limit = " + low95 + ", upper limit = " + high95);
            System.out.println("    99% confidence lower limit = " + low99 + ", upper limit = " + high99);
            
            //Scaling
            for (int rawNdx = 0; rawNdx < fCW.getRowCounts(); rawNdx++){
                if (outputFlag[rawNdx]){
                    if (confidenceChk){                                                 //Confidence check and trim
                        if (pctConfidence == 95){                                           //95% Confidence Interval
                            if (ratios[rawNdx] > low95 && ratios[rawNdx] < high95){             //Within 95% confidence range
                                if (Parameter.pRefStr.equalsIgnoreCase("Cy3")){                 //Scale Cy5
                                    cy5Column[rawNdx] = cy3Column[rawNdx] * ratios[rawNdx];
                                }else{                                                              //Scale Cy3
                                    cy3Column[rawNdx] = cy5Column[rawNdx] / ratios[rawNdx];
                                }
                                confidentCounts++;
                            }else{
                                outputFlag[rawNdx] = false;
                            }
                        }else{                                                              //99% Confidence Interval
                            if (ratios[rawNdx] > low99 && ratios[rawNdx] < high99){             //Within 95% confidence range
                                if (ref.equalsIgnoreCase("Cy3")){                 //Scale Cy5
                                    cy5Column[rawNdx] = cy3Column[rawNdx] * ratios[rawNdx];
                                }else{                                                              //Scale Cy3
                                    cy3Column[rawNdx] = cy5Column[rawNdx] / ratios[rawNdx];
                                }
                                confidentCounts++;
                            }else{
                                outputFlag[rawNdx] = false;
                            }
                        }
                    }else{                                                              //No Confidence check and trim
                        if (Parameter.pRefStr.equalsIgnoreCase("Cy3")){                 //Scale Cy5
                            cy5Column[rawNdx] = cy3Column[rawNdx] * ratios[rawNdx];
                        }else{                                                              //Scale Cy3
                            cy3Column[rawNdx] = cy5Column[rawNdx] / ratios[rawNdx];
                        }
                    }
                }else{
                    cy3Column[rawNdx] = 0;
                    cy5Column[rawNdx] = 0;
                }
            }
            
            if (confidenceChk){
                System.out.println("    Gene counts within confidence range = " + confidentCounts);
            }
            
            fCW.setColOneArray(cy3Column);
            fCW.setColTwoArray(cy5Column);
            fCW.setRowFlag(outputFlag);
            
            
            //Output post-normalization RI results
            
            /* Mod for MeV
            String postPrcName = super.proj.getProjPath() + "ratio_stats" + FileBrowser.fsep + fileCW.getFileName() + "_post.prc";
            dataPrep = new DataPrep(fileCW);
            dr = new DataReporter(postPrcName, dataPrep.getLogProd(), dataPrep.getLogRatio(), new IndexAdjuster().addOne(dataPrep.getRawNdx()));
            int outCount = dataPrep.getNonZeroCount();
        
            Hashtable hash = new Hashtable();
            hash.put(new String("pre_prc_name"), prePrcName);
            hash.put(new String("post_prc_name"), postPrcName);
            hash.put(new String("confident_count"), new Integer(confidentCounts));
            hash.put(new String("out_row_count"), new Integer(outCount));
            proj.insertReportItem("ratio_stats_for_" + fileCW.getFileName(), hash);
            */
            
        }catch (Exception ex){
            throw ex;
        }        
    }
    
    public final ColumnWorker getRatioStatsColumnWorker(){
        return fCW;
    }
    
    public final float getPlainMean(){
        return plainMean;
    }
    
    public final float getCalcMean(){
        return calcMean;
    }
    
    public final float getCalcSD(){
        return calcSD;
    }
    
    public final float getLow95(){
        return low95;
    }
    
    public final float getHigh95(){
        return high95;
    }
    
    public final float getLow99(){
        return low95;
    }
    
    public final float getHigh99(){
        return high95;
    }
    
    public final float getCval(){
        return cVal;
    }
    
    public final int getIterationCounts(){
        return iterCount;
    }
    
    public final int getValidRatioCounts(){
        return validRatioCounts;
    }
    
    public final int getConfidentCounts(){
        return confidentCounts;
    }

    public static void main(String args[]) {
        try{
            ColumnWorker fileCW = new ColumnWorker("C:\\DataFiles0\\32K\\SimonData\\1233259D2F.tav");
            fileCW.setColumnsForMeta(2, 3);
            fileCW.setColumns(6, 7, 15, 16);
            fileCW.computeGoodRatiosAndLogRatios(0, 0);
            RatioStats ratioStats = new RatioStats(fileCW, true, 99, "Cy3");
        }catch (FileNotFoundException nfex){
            nfex.printStackTrace();
        }catch (NumberFormatException nex){
            nex.printStackTrace();
        }catch (Exception ex){
            ex.printStackTrace();
        }
    }
    
}
